#include "ados.h"
#include "standard.h"

/************************************************************************
 * NAME:	vtoc_create()
 *
 * DESCR:	Creates a new vtoc structure and allocates it.
 *
 * ARGS:	
 *
 * RETURNS:	a pointer to the vtoc if OK, NULL otherwise.
 *
 * NOTES:	
 ************************************************************************/
ados_vtoc *
ados_vtoc_create(int track, int sectors)
{
    ados_vtoc	*newvtoc;

    if ((newvtoc = (ados_vtoc *)malloc(sizeof(ados_vtoc))) != NULL) {
	newvtoc->allocation = (int *)malloc(track*sectors*sizeof(int));
	if (newvtoc->allocation == NULL) {
	    free(newvtoc);
	    newvtoc = NULL;
	}
    }

    return(newvtoc);
}

/************************************************************************
 * NAME:	vtoc_destroy()
 *
 * DESCR:	Destroys the given vtoc.
 *
 * ARGS:	
 *
 * RETURNS:	nothing
 *
 * NOTES:	
 ************************************************************************/
void
ados_vtoc_destroy(ados_vtoc *vtoc)
{
    free(vtoc->allocation);
    free(vtoc);
}

/************************************************************************
 * NAME:	ados_allocation_print()
 *
 * DESCR:	DEBUG routine to print out the allocation table using M().
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
ados_allocation_print(struct adosfs *adosfs)
{
    int	track, sector;
    int	tracks, sectors;

    tracks = adosfs->vtoc->tracks;
    sectors = adosfs->vtoc->sectors;

    M("    0 1 2 3 4 5 6 7 8 9 A B C D E F\n");
    for (track=0; track < tracks; track++) {
	M1("%2d: ", track);
	for (sector=0; sector < sectors; sector++) {
	    M1("%1d ",ALLOCATION(adosfs->vtoc,track,sector));
	}
	M("\n");
    }
}

/************************************************************************
 * NAME:	ados_allocation_parse()
 *
 * DESCR:	Parses allocation bytes into the allocation array.
 *
 * ARGS:	data - pointer to up to 4 bytes of incoming allocation bitmap
 *			which is assumed to represent an entire track of sectors
 *		allocation - pointer to array of "sectors" of ints
 *		sectors - number of sectors to parse in the data
 *
 * RETURNS:	
 *
 * NOTES:	- calculations assume that there will never be zero sectors
 *
 * this is a little weird, and I'm not quite sure how to do it
 * the problem is that "BAD" defined the table ONLY for 16-sector
 * images.  I'm not sure how to handle fewer sectors.  So I'm
 * assuming that the bits are in direct order in the bit map
 * assuming 4 contiguous bytes:				
 *								
 *     0x38      0x39      0x3a      0x3b			
 *  xxxx xxxx xxxx xxxx xxxx xxxx xxxx xxxx			
 *  FEDC BA98 7654 3210 ---- ---- ---- ----  16 sector disk	
 *  CBA9 8765 4321 0--- ---- ---- ---- ----  13 sector disk	
 ************************************************************************/
static void
ados_allocation_parse(unsigned char *data, int *allocation, int sectors)
{
    int	bytes = (sectors-1) / 8 + 1;
    int	apos = sectors-1;
    int	mask;
    int i;

    for (i=0; i < bytes; i++) {
	for (mask=0x80; mask != 0 && apos >= 0; mask = mask >> 1, apos--) {
	    allocation[apos] = (data[i]&mask)?FALSE:TRUE;
	}
    }
}

static void
ados_allocation_unparse(unsigned char *data, int *allocation, int sectors)
{
    int	bytes = (sectors-1) / 8 + 1;
    int	apos = sectors-1;
    int	mask;
    int i;

    for (i=0; i < bytes; i++) {
	data[i] = 0;
	for (mask=0x80; mask != 0 && apos >= 0; mask = mask >> 1, apos--) {
	    data[i] |= (allocation[apos])?0:mask;
	}
    }
    
}

/************************************************************************
 * NAME:	ados_vtoc_free_count()
 *
 * DESCR:	Returns the count of free blocks according to the
 *		allocation table.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
ados_vtoc_free_count(ados_vtoc *vtoc)
{
    int	total = 0;
    int track, sector;
    int tracks, sectors;

    tracks = vtoc->tracks;
    sectors = vtoc->sectors;

    for (track=0; track < tracks; track++) {
	for (sector=0; sector < sectors; sector++) {
	    total += (ALLOCATED(vtoc,track,sector))?0:1;
	}
    }

    return(total);
}

/************************************************************************
 * NAME:	vtoc_read() & vtoc_write()
 *
 * DESCR:	Reads/Writes in a vtoc from the given floppy image, and returns
 *		a new vtoc structure with the data.
 *
 * ARGS:	
 *
 * RETURNS:	vtoc_read() - NULL if bad read or out of memory, otherwise
 *		a vtoc (which should be freed when done)
 *		vtoc_write() - TRUE if the write was done, FALSE otherwise
 *
 * NOTES:	- yikes - look at the allocation table stuff!
 ************************************************************************/
ados_vtoc *
ados_vtoc_read(struct adosfs *adosfs)
{
    ados_vtoc		*vtoc;
    unsigned char	 data[VTOC_SIZE];
    int			 phys_sector;
    int			 tracks, sectors;

    /* get the data for the VTOC sector	*/

    if (!ados_getlsect(adosfs, VTOC_TRACK, VTOC_SECTOR, data)) {
	return(NULL);
    }

    tracks = (int) data[0x34];
    sectors = (int) data[0x35];

    /* although the floppy structure already has these set, get them	*/
    /* from the VTOC for the filesystem itself.				*/

    if (tracks < VTOC_SECTOR || sectors == 0) {
	return(NULL);
    }

    if ((vtoc = ados_vtoc_create(tracks,sectors)) == NULL) {
	return(NULL);
    }

    vtoc->tracks = tracks;
    vtoc->sectors = sectors;

    vtoc->catalog.track = (int)data[0x01];
    vtoc->catalog.sector = (int)data[0x02];
    vtoc->release = (int)data[0x03];
    vtoc->volume = (int)data[0x06];
    vtoc->max_ts = (int)data[0x27];
    vtoc->last_track = (int)data[0x30];
    vtoc->direction = (int)data[0x31];
    vtoc->sector_size = int_read(&(data[0x36]));
    vtoc->allocation = (int *)malloc(tracks*sectors*sizeof(int));
    if (vtoc->allocation == NULL) {
	ados_vtoc_destroy(vtoc);
	return(NULL);
    }

    {    /* time to read in the allocation table -------------------------	*/

	int	track, pos;

	for (pos=0x38, track=0; track < tracks; track++, pos += 4) {
	    ados_allocation_parse(data+pos, vtoc->allocation + (track*sectors), sectors);
	}
    }

    return(vtoc);
}

int
ados_vtoc_write(struct adosfs *adosfs)
{
    ados_vtoc		*vtoc;
    unsigned char	 data[VTOC_SIZE];
    int			 phys_sector;

    vtoc = adosfs->vtoc;

    /* set all bytes to zero first	*/

    {
	int i;
	for (i=0; i < VTOC_SIZE; i++) {
	    data[i] = (unsigned char)0;
	}
    }

    data[0x01] = (unsigned char) vtoc->catalog.track;
    data[0x02] = (unsigned char) vtoc->catalog.sector;
    data[0x03] = (unsigned char) vtoc->release;
    data[0x06] = (unsigned char) vtoc->volume;
    data[0x27] = (unsigned char) vtoc->max_ts;
    data[0x30] = (unsigned char) vtoc->last_track;
    data[0x31] = (unsigned char) vtoc->direction;
    data[0x34] = (unsigned char) vtoc->tracks;
    data[0x35] = (unsigned char) vtoc->sectors;
    int_write(vtoc->sector_size, &(data[0x36]));

    {    /* time to fill in the allocation table -------------------------	*/

	int	track, pos;

	for (pos=0x38, track=0; track < vtoc->tracks; track++, pos += 4) {
	    ados_allocation_unparse(data+pos, vtoc->allocation + (track*vtoc->sectors), vtoc->sectors);
	}
    }

    /* get the data for the VTOC sector	*/

    if (!ados_putlsect(adosfs, VTOC_TRACK, VTOC_SECTOR, data)) {
	return(FALSE);
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	ados_vtoc_sector_allocate()
 *
 * DESCR:	Allocates a new sector from the VTOC.  Fills in the given
 *		t/s reference with the address of the new sector.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if there was space left, FALSE otherwise.
 *
 * NOTES:	- uses the following allocation algorithm:
 *		  - moves out from the directory track to higher numbered
 *		    tracks and sectors from 0 to 16
 *		  - then moves in from the directory track to lower
 *		    numbered tracks and sectors from 0 to 16
 ************************************************************************/
int
ados_vtoc_sector_allocate(struct adosfs *adosfs, ados_ts_ref *ts)
{
    int		i,j;
    for (i=VTOC_TRACK; i < adosfs->vtoc->tracks; i++) {
	for (j=0; j < adosfs->vtoc->sectors; j++) {
	    if (!ALLOCATED(adosfs->vtoc,i,j)) {
		ts->track = i;
		ts->sector = j;
		ALLOCATION_SET(adosfs->vtoc,i,j);
		return(TRUE);
	    }
	}
    }
    return(FALSE);
}

/************************************************************************
 * NAME:	ados_vtoc_sector_free()
 *
 * DESCR:	Frees the given sector.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the sector was previously in use, FALSE if it
 *		looks like the sector was already free.
 *
 * NOTES:	
 ************************************************************************/
int
ados_vtoc_sector_free(struct adosfs *adosfs, ados_ts_ref *ts)
{
    int		ret = ALLOCATION_SET(adosfs->vtoc,ts->track,ts->sector)?TRUE:FALSE;

    if (ts->track != 0) {
	ALLOCATION_CLEAR(adosfs->vtoc,ts->track,ts->sector);
    }

    return(ret);
}

